#!/bin/bash

# Usage: pydelphi_cpu_job.sh <gridtype> <ncpu> <diel_model>
gridtype="$1"       # "cubic" or "cuboidal"
ncpu="$2"           # number of CPU threads
diel_model="$3"     # "trad", "gauss", or "gausscut"
runid="$4"

if [[ -z "$gridtype" || -z "$ncpu" || -z "$diel_model" ]]; then
    echo "Usage: $0 <gridtype> <ncpu> <diel_model>"
    exit 1
fi

# Fixed gridbox param by gridtype
if [[ "$gridtype" == "cubic" ]]; then
    boxtype="cbc"
    gridboxparam="pf70"
elif [[ "$gridtype" == "cuboidal" ]]; then
    boxtype="cbl"
    gridboxparam="bxm15"
else
    echo "Error: Unknown gridtype '$gridtype'. Allowed: cubic, cuboidal"
    exit 2
fi

# Diel model template prefix and short name for job tag
case "$diel_model" in
    trad)
        diel_tag="D2"
        template_prefix="trad"
        ;;
    gauss)
        diel_tag="Gsn"
        template_prefix="gauss"
        ;;
    gausscut)
        diel_tag="Gsc"
        template_prefix="gausscut"
        ;;
    *)
        echo "Error: Unknown dielectric model '$diel_model'. Use trad, gauss, or gausscut"
        exit 3
        ;;
esac

template_file="prm.templates/${template_prefix}-${gridtype}.prm"


# --- Directory Check and Creation (with warning, then proceed) ---
if [ -d "$runid" ]; then
    echo "Warning: Directory '$runid' already exists. Assuming it belongs to the same batch of runs and proceeding."
else
    echo "Directory '$runid' does not exist. Creating it..."
    mkdir -p "$runid" # -p creates parent directories if they don't exist
    if [ $? -ne 0 ]; then
        echo "Error: Failed to create directory '$runid'. Aborting."
        exit 1 # Still abort if directory creation itself fails (e.g., permissions)
    fi
fi
# --- End Directory Check ---

module load anaconda3/2023.09-0
module load cuda/12.3.0  # can keep this even if no GPU used
source $(conda info --base)/etc/profile.d/conda.sh
conda activate py312_pydelphi020


for pqr in $(cat ../pqr-file-list); do
    PDB=${pqr/.pqr/}
    pdb=${PDB,,}
    pdb=${pdb//_/-}
    echo "Processing $pdb with ${ncpu} threads, ${gridboxparam}, dielectric=${diel_model}"

    foutprefix="${runid}/${pdb}_trad-${boxtype}-${gridboxparam}-cpu${ncpu}"
    prmfile="${foutprefix}.prm"

    lscpu > ${foutprefix}-lscpu.txt
    cat /proc/cpuinfo > ${foutprefix}-cpuinfo.txt

    sed "s/__PQR_FILE__/${pqr}/" "$template_file" > "$prmfile"

    /usr/bin/time -v pydelphi-static -p double -P cpu -t "$ncpu" -f "$prmfile" \
        -o "${foutprefix}-f8.csv" -l "$pdb" -v info -S -O \
        > "${foutprefix}-f8.log" 2>&1
done


for pqr in $(cat ../pqr-file-list); do
    PDB=${pqr/.pqr/}
    pdb=${PDB,,}
    pdb=${pdb//_/-}
    echo "Processing $pdb with ${ncpu} threads, ${gridboxparam}, dielectric=${diel_model}"

    foutprefix="${runid}/${pdb}_trad-${boxtype}-${gridboxparam}-cpu${ncpu}"
    prmfile="${foutprefix}.prm"

    lscpu > ${foutprefix}-lscpu.txt
    cat /proc/cpuinfo > ${foutprefix}-cpuinfo.txt

    sed "s/__PQR_FILE__/${pqr}/" "$template_file" > "$prmfile"

    /usr/bin/time -v pydelphi-static -p single -P cpu -t "$ncpu" -f "$prmfile" \
        -o "${foutprefix}-f4.csv" -l "$pdb" -v info -S -O \
        > "${foutprefix}-f4.log" 2>&1

done

